/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：入力PDFをPDF/A-1bまたはPDF/A-2bに変換する。
        変換の際にリニアライズして保存するかを選択する。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class FixUpPDFASetSaveOption {

    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.print("usage: java FixUpPDFA in-pdf-file out-pdf-file");
        System.out.println(" PDF/Aの種類 保存時のオプション\n");
        System.out.println("PDF/Aの種類");
        System.out.println("1 : PDF/A-1b  2 : PDF/A-2b");
        System.out.println("保存時のオプション");
        System.out.println("0 : デフォルト  1 : リニアライズする");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 4) {
            printUsage();
            return;
        }

        int kind;
        try {
            kind =  Integer.parseInt(args[2]);
            switch (kind) {
            case 1:
            case 2:
                break;
            default:
                System.out.println("PDF-Aの種類には、1または2の数値を指定してください。");
                printUsage(); // usageメッセージの表示
                return;
            }
        }
        catch(NumberFormatException numfe) {
            System.out.println("PDF-Aの種類には、1または2の数値を指定してください。");
            printUsage(); // usageメッセージの表示
            return;
        }
        
        boolean setLinearize;
        try {
            setLinearize = readBoolArgs(args[3],
                                  "保存時のオプションには、0または1の数値を指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }

        
        // コマンドライン引数の取得
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFFixUp fixUp = new PtlPDFFixUp()) {
            
            if(setLinearize){
                // リニアライズする
                fixUp.setSaveOption(PtlPDFFixUp.SAVE_OPTION.SAVE_LINEARIZE);
            }else{
                //デフォルト
                fixUp.setSaveOption(PtlPDFFixUp.SAVE_OPTION.SAVE_RECONSTRUCT);
            }
            
            // PDFFixUPでinputFileをPDF/Aにする。
            switch (kind) {
            case 1: //PDF/A-1b
                fixUp.fixUpPDFA(PtlPDFFixUp.PDFA_TYPE.PDFA_1B, inputFile);
                break;
            case 2:
                // PDF/A-2b
                fixUp.fixUpPDFA(PtlPDFFixUp.PDFA_TYPE.PDFA_2B, inputFile);
                break;
            }
            
            //変換エラー発生チェック
            try(PtlPDFFixUpErrors fixUpErrors = fixUp.getErrors()) {
                int numErrors = fixUpErrors.getCount();
                System.out.println("変換時の総エラー数 : " + numErrors);
                
                if(numErrors == 0){
                    System.out.println("エラーは発生しませんでした。");
                }else {
                    for(int i = 0; i < numErrors; i++) {

                        try(PtlPDFFixUpError thisFixUpError = fixUpErrors.get(i)){
                            System.out.println("エラーその"+ i +"："+ thisFixUpError.getErrorMessageJP());
                        }
                    }
                }
            }
            
            // ファイルに保存します。
            fixUp.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
    
    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
        throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }
}