/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：文書情報の取得

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class GetDocInfoWithFontInfo {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java GetDocInfoWithFontInfo in-pdf-file");
     }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 1) {
            printUsage();
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロード
            doc.load(inputFile);

            try (PtlDocProperty docProperty = doc.getDocProperty(); //PDFの文書プロパティ
                 PtlDocInfo docinf = docProperty.getDocInfo();    //PDFの文書情報
                 // PDFのカスタムプロパティ
                 PtlCustomProperties customProperties = docProperty.getCustomProperties();
                 //フォント情報を扱うコンテナ
                 PtlFontInfos fontInfos =  docProperty.getFontInfos()   ) {
                //フォント情報を取得
                getDocFontInfo(fontInfos);
                
                // タイトル取得
                System.out.println("Title : " + docinf.getTitle());
                // 著者取得
                System.out.println("Author : " + docinf.getAuthor());
                // サブジェクト取得
                System.out.println("Subject : " + docinf.getSubject());
                // キーワード取得
                System.out.println("Keywords : " + docinf.getKeywords());
                // クリエータ取得
                System.out.println("Creator : " + docinf.getCreator());
                // プロデューサ取得
                System.out.println("Producer : " + docinf.getProducer());
                // 作成日付を取得
                try (PtlDate dateCreate = docinf.getCreationDate()) {
                    System.out.println("CreationDate : "
                        + dateCreate.getYear()
                        + "/" + dateCreate.getMonth()
                        + "/" + dateCreate.getDay()
                        + " " + dateCreate.getHour()
                        + ":" + dateCreate.getMin()
                        + ":" + dateCreate.getSec());
                }

                // 更新日付を取得
                try (PtlDate dateMod = docinf.getModDate()) {
                    System.out.println("ModDate : "
                        + dateMod.getYear()
                        + "/" + dateMod.getMonth()
                        + "/" + dateMod.getDay()
                        + " " + dateMod.getHour()
                        + ":" + dateMod.getMin()
                        + ":" + dateMod.getSec());
                }
                
                // カスタムプロパティが空かどうか判定
                System.out.println("");
                if(customProperties.isEmpty()) {
                    //空の場合はそう表示する
                    System.out.println("このPDFにカスタムプロパティは設定されていません。");
                }else {  // カスタムプロパティが空でない場合に内容を取得する
                    System.out.println("このPDFのカスタムプロパティを表示します。");
                    int propertiesNum = customProperties.getCount();
                    for(int i = 0; i < propertiesNum; i++){
                        System.out.println("Custom Property No." + (i+1) +":");
                        try(PtlCustomProperty customProperty = customProperties.get(i)){
                            System.out.println("名前 : " + customProperty.getName());
                            System.out.println("値 : " + customProperty.getValue());
                            System.out.println("");
                        }
                    }
                }
            }
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
    
    
    /**
     * ドキュメントが持つフォント情報を全て表示する関数。
     * @param fontInfos
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getDocFontInfo(PtlFontInfos fontInfos)
    throws PtlException, Exception, Error {
        System.out.println("Font : ");
        //フォント情報の有無を確認する
        if(fontInfos.isEmpty()) {
            System.out.println("  このドキュメントにフォント情報はありません。");
            
        }else {
            //フォント情報がある場合の処理を行う
            int fontNum = fontInfos.getCount();
            System.out.println("  フォント総数 : " + fontNum);
            for(int i = 0; i < fontNum; i++) {
                try(PtlFontInfo fontInfo = fontInfos.get(i)) {
                    System.out.println("  --Font index No: " + i);
                    System.out.println("  フォント名：" + fontInfo.getFontName());
                    System.out.println("  フォントタイプ：" +  fontInfo.getFontType());
                    System.out.println("  エンコーディング名：" + fontInfo.getEncodingName());
                    System.out.println("  エンコーディングタイプ：" + fontInfo.getEncodingType());
                    if(fontInfo.isEmbedded()) {
                        System.out.println("  フォント埋め込み：あり");
                    }else {
                        System.out.println("  フォント埋め込み：なし");
                    }
                }
            }
        }
    }
}