/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：入力PDFに含まれている全てのページの全ての画像について、
        画像に設定されているマスク情報を抜き出します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;


public class GetImageMaskType {
    
    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java GetImageMaskType in-pdf-file");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 1) {
            printUsage();
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            try (PtlPages pages = doc.getPages()) { //ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("入力PDFのページコンテナが空");
                    return;
                }

                // それ以外であれば指定ページ番号を検索。
                int wholePageNum = doc.getPageCount();
                for(int i = 0; i < wholePageNum; i++) {
                    // ページの取得(index番号は0が先頭のため1を引く)
                    try (PtlPage page = pages.get(i)) {
                        System.out.println("このPDFの"+
                                        i +"ページ目の画像の情報を読み取ります。");
                        // 画像情報の読み取り
                        getImageMaskType(page);
                    }
                }
            }
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void getImageMaskType(PtlPage page)
        throws PtlException, Exception, Error {
        // ページコンテント・画像エレメントの取得
        try (PtlContent content = page.getContent();
             PtlEditElements elems = content.getEditElements(PtlContent.GET_IMAGE)) {
            int numElems = elems.getCount();
            if (numElems == 0) {
                System.out.println("指定されたページに画像エレメントがありませんでした。");
                return; // 画像エレメントが無い場合はreturnする
            }
            //imageIndexNumのエラー処理


            // 画像エレメントの取得
            for(int i = 0; i < numElems; i++){
                System.out.println("画像のインデックス番号：" + i);
                
                //各エレメントの処理
                try (PtlEditElement elem = elems.get(i)) {                    
                    boolean isImage = false;
                    //エレメントが画像であるかの判定
                    switch (elem.getType()) {
                    case TYPE_IMAGE:
                        isImage = true;
                        break;
                    default:
                        System.out.println("指定されたエレメントは画像ではありません。");
                        break;
                    }
                    
                    if(isImage) {
                        try(PtlEditImage elemImage = (PtlEditImage)elem) {
                            switch (elemImage.getMaskType()) {
                            case MASK_COLORKEY:
                                System.out.println("この画像にはカラーキーマスクが設定されています。");
                                break;
                            case MASK_EXPLICIT:
                                System.out.println("この画像には明示マスクが設定されています。");
                                break;
                            case MASK_NONE:
                                System.out.println("この画像にはマスクが設定されていません。");
                                break;
                            case MASK_SOFT:
                                System.out.println("この画像にはソフトマスクが設定されています。");
                                break;
                            case MASK_STENCIL:
                                System.out.println("この画像にはステンシルマスクが設定されています。");
                                break;
                            //どのマスクタイプでもなかった場合の処理
                            default:
                                System.out.println("マスクのタイプ情報を読み取れませんでした。");
                                break;
                            }
                        }
                    }
                }
            }
        }
    }
}