/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：Contentに含まれるFormXOnjectエレメントの情報を表示します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class GetPtlEditFormXObjectInfo {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java GetPtlEditFormXObjectInfo in-pdf-file");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 1) {
            printUsage();
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            try (PtlPages pages = doc.getPages()) { //ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("入力PDFのページコンテナが空");
                    return;
                }
                
                for(int i = 0;i < pages.getCount();i++) { // ページ数の取得
                    System.out.println("-----------------");
                    System.out.println(i+"ページ目の情報を表示します。");
                    try (PtlPage page = pages.get(i)) {
                        getFormXObjectInfoInPage(page);
                    }
                    System.out.println("-----------------");
                }
            }
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    private static void getFormXObjectInfoInPage(PtlPage page)
                        throws PtlException, Exception, Error {
        try(PtlContent content = page.getContent();// 挿入先ページコンテントの取得
            PtlEditElements elems = content.getEditElements()) {
            
            int numOfElements = elems.getCount();
            System.out.println("このページに含まれるエレメントの数は" + numOfElements + "です。");
            for(int i = 0; i < numOfElements; i++) {
                try(PtlEditElement element = elems.get(i)) {
                    if(element.getType() == PtlEditElement.ELEMENT_TYPE.TYPE_FORMXOBJECT) {
                        System.out.println("このページの" + i 
                                            + " 番目のエレメントがForm XObjectでした。内容を確認します。");
                        try(PtlEditFormXObject formXObject = (PtlEditFormXObject)element;
                            PtlContent contentInXObject = formXObject.getContent();// FormXObjectに含まれる
                            PtlEditElements elementsInXObject = contentInXObject.getEditElements()) {                        
                            getFormXObjectInfo(elementsInXObject);
                        }
                    }
                }
            }
        }
    }
    
    private static void getFormXObjectInfo(PtlEditElements elems)
                        throws PtlException, Exception, Error {
        int numOfElements = elems.getCount();
        System.out.println("このFormXObjectに含まれるエレメントの数は" + numOfElements + "です。");


        for(int i = 0; i < numOfElements; i++) {

            System.out.println("---このFormXObjectの" + i + "番目のエレメントの内容を確認します。---");
            try(PtlEditElement element = elems.get(i)) {
                //エレメントに共通の情報（BBox, ClippedBox）の情報を表示
                getElementInfo(element);
                
                switch(element.getType()) {
                    case TYPE_FORMXOBJECT:
                        System.out.println("このエレメントは入れ子になったForm XObjectでした。内容を確認します。");
                        try(PtlEditFormXObject formXObject = (PtlEditFormXObject)element;
                            PtlContent contentInXObject = formXObject.getContent();// FormXObjectに含まれるContentを取得
                            PtlEditElements elementsInXObject = contentInXObject.getEditElements()) {
                            System.out.println("--"); 
                            getFormXObjectInfo(elementsInXObject);
                        }
                        break;
                        
                    case TYPE_TEXT:
                        getEditTextInfo(element);
                        break;
                        
                    case TYPE_PATH:
                        getEditPathInfo(element);
                        break;

                    case TYPE_IMAGE:
                        getEditImageInfo(element);
                        break;
                }
            }
        }
    }
    

    private static void getElementInfo(PtlEditElement element)
                        throws PtlException, Exception, Error {
        System.out.println("エレメントのBBoxの情報を表示します。");
        //BBoxの情報を印字
        try(PtlRect bBox = element.getBBox()) {
            getRectInfo(bBox, "このエレメントのBBox");
        }
        
        //ClippedBoxの情報を印字
        if(element.hasClip()){
            try(PtlRect clippedBox = element.getClippedBBox()) {
                System.out.println("このエレメントにはクリッピングがあります。");
                getRectInfo(clippedBox, "このエレメントのClippedBox");
            }
        }
    }    

    private static void getEditPathInfo(PtlEditElement element)
                        throws PtlException, Exception, Error {
        try(PtlEditPath pathElement = (PtlEditPath)element;
            PtlEditPathItems pathItems = pathElement.getPathItems()) {

            System.out.println("PATHエレメント固有の情報を表示します。");
            System.out.println("このPATHエレメントのペイントフラグ情報を表示します。");
            System.out.println("-");
            System.out.print("ペイントフラグ：");
            int paintFlag = pathElement.getPaintFlags();
            switch(paintFlag) {
                case PtlEditPath.PAINT_EOFILL: //パスを塗りつぶします。奇偶規則。
                    System.out.println("奇偶規則に基づいてパスを塗りつぶす");
                    break;

                case PtlEditPath.PAINT_FILL: //パスを塗りつぶします。非ゼロ回転規則。
                    System.out.println("非ゼロ回転規則に基づいてパスを塗りつぶす");
                    break;

                case PtlEditPath.PAINT_INVISIBLE: //ストロークも塗りつぶしも行いません。
                    System.out.println("ストロークも塗りつぶしも行わない");
                    break;

                case PtlEditPath.PAINT_STROKE: //ストロークします。
                    System.out.println("ストロークする");
                    break;

                default:
                    System.out.println("不明なフラグが取得されました。");
                    System.out.println("paintFlag = " + paintFlag);
                    break;
            }

            //ストロークカラー情報の取得
            getStrokeColorInfo(pathElement);
            //塗りつぶしカラー情報の取得
            getFillColorInfo(pathElement);
            //パスアイテムの情報取得
            getPathItemsInfo(pathItems);                    
        }
    }

    private static void getPathItemsInfo(PtlEditPathItems pathItems)
                        throws PtlException, Exception, Error {
        System.out.println("-");
        System.out.println("パスアイテムの情報を表示します。");
        for(int j = 0; j < pathItems.getCount(); j++) {
            try(PtlEditPathItem item = pathItems.get(j)) {
                PtlEditPathItem.PATH_ITEM_TYPE itemType = item.getType();

                System.out.println( j + "番目のアイテム情報：");
                System.out.println("---");
                switch(itemType) {
                    case TYPE_PATH_CURVE:
                        System.out.println("アイテム種別：ベジェ曲線");
                        try(PtlEditPathCurve pathCurve = (PtlEditPathCurve)item) {
                            getPathCurveInfo(pathCurve);
                        }
                        break;

                    case TYPE_PATH_LINE:
                        System.out.println("アイテム種別：直線");
                        try(PtlEditPathLine pathLine = (PtlEditPathLine)item) {
                            getPathLineInfo(pathLine);
                        }
                        break;

                    case TYPE_PATH_RECT:
                        System.out.println("アイテム種別：矩形");
                        try(PtlEditPathRect pathRect = (PtlEditPathRect)item;
                            PtlRect rectOfPath = pathRect.getRect()) {
                            getRectInfo(rectOfPath, "このパスアイテムの矩形");
                        }
                        break;
                }
                System.out.println("---");
            }
        }
    }

    private static void getPathCurveInfo(PtlEditPathCurve pathCurve)
                        throws PtlException, Exception, Error {
        try(PtlPoint startPoint    = pathCurve.getStartPoint();
            PtlPoint endPoint      = pathCurve.getEndPoint();
            PtlPoint controlPoint1 = pathCurve.getControlPoint1();
            PtlPoint controlPoint2 = pathCurve.getControlPoint2()) {
            System.out.print("開始位置[x, y]: ");
            System.out.println("[" + startPoint.getX() + ", " + startPoint.getY() + "]");
            System.out.print("終了位置[x, y]: ");
            System.out.println("[" + endPoint.getX() + ", " + endPoint.getY() + "]");
            System.out.print("第一の制御点[x, y]: ");
            System.out.println("[" + controlPoint1.getX() + ", " + controlPoint1.getY() + "]");
            System.out.print("第二の制御点[x, y]: ");
            System.out.println("[" + controlPoint2.getX() + ", " + controlPoint2.getY() + "]");
        }
    }
    
    private static void getPathLineInfo(PtlEditPathLine pathLine)
                        throws PtlException, Exception, Error {
        try(PtlPoint startPoint = pathLine.getStartPoint();
            PtlPoint endPoint   = pathLine.getEndPoint()) {
            System.out.print("開始位置[x, y]: ");
            System.out.println("[" + startPoint.getX() + ", " + startPoint.getY() + "]");
            System.out.print("終了位置[x, y]: ");
            System.out.println("[" + endPoint.getX() + ", " + endPoint.getY() + "]");
        }
    }
    
    /**
     * 読み込まれたページに含まれるTEXTエレメントが持つ情報を表示する関数です。
     * @param page 読み込まれたページ
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getEditTextInfo(PtlEditElement element)
                        throws PtlException, Exception, Error {
        try(PtlEditText textElement = (PtlEditText)element;
            PtlEditTextItems textItems = textElement.getTextItems()) {
            System.out.println("TEXTエレメント固有の情報を表示します。");
            System.out.println("テキストアイテムの情報を表示します。");
            getTextItemsInfo(textItems);
        }
    }

    /**
     * Textアイテムが持つ情報を表示する関数です
     * @param textItems そのエレメントが持つTextItemsコンテナクラス
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getTextItemsInfo(PtlEditTextItems textItems)
                        throws PtlException, Exception, Error {
        int numOfTextItems = textItems.getCount();
        System.out.println("このテキストアイテムに含まれるアイテム数は" + numOfTextItems + "です。");
        for(int j = 0; j < numOfTextItems; j++) {
            System.out.println( j + "番目のアイテム情報：");
            System.out.println("---");

            try(PtlEditTextItem item = textItems.get(j);
                PtlRect textItemBBox =item.getBBox();
                PtlFontInfo fontInfo = item.getFontInfo()) {
                System.out.println("テキストアイテムのBBox情報：");
                getRectInfo(textItemBBox, "このテキストアイテムのBBox");

                System.out.println("-");
                System.out.println("テキストアイテムが含むテキスト：");
                System.out.println(item.getText());

                System.out.println("-");
                System.out.println("フォント情報：");
                System.out.println("フォント名：" + fontInfo.getFontName());
                System.out.println("フォントタイプ：" +  fontInfo.getFontType());
                System.out.println("エンコーディングタイプ：" + fontInfo.getEncodingType());
                if(fontInfo.getEncodingType()==PtlFontInfo.ENCODING_TYPE.ENCODING_NAME) {
                    System.out.println("エンコーディング名：" + fontInfo.getEncodingName());
                }
                if(fontInfo.isEmbedded()) {
                    System.out.println("フォント埋め込み：あり");
                }else {
                    System.out.println("フォント埋め込み：なし");
                }

                System.out.println("-");
                System.out.print("ペイントフラグ：");
                int paintFlag = item.getPaintFlags();
                switch(paintFlag) {
                    case PtlEditPath.PAINT_FILL: //パスを塗りつぶす
                        System.out.println("パスを塗りつぶす");
                        System.out.println("塗りつぶしカラー情報を取得します。");
                        getFillColorInfo(item); //塗りつぶしカラー情報を取得するための関数
                        break;

                    case PtlEditPath.PAINT_INVISIBLE: //ストロークも塗りつぶしも行いません。
                        System.out.println("ストロークも塗りつぶしも行わない");
                        break;

                    case PtlEditPath.PAINT_STROKE: //ストロークする
                        System.out.println("ストロークする");
                        System.out.println("ストロークカラー情報を取得します");
                        getStrokeColorInfo(item); //ストロークカラー情報を取得するための関数
                        break;

                    default:
                        System.out.println("不明なフラグが取得されました。");
                        System.out.println("paintFlag = " + paintFlag);
                        break;
                }
                System.out.println("---");
            }
        }
    }
    
    public static void getEditImageInfo(PtlEditElement element)
        throws PtlException, Exception, Error {
        // ページコンテント・画像エレメントの取得
        // OUTPUT_FORMATはFORMAT_AUTOで設定。
        // 画像エレメントの取得
        try(PtlEditImage elemImage = (PtlEditImage)element) {
            System.out.println("画像の高さ：" + elemImage.getHeight());
            System.out.println("画像の幅　：" + elemImage.getWidth());
            System.out.println("画像のPPI ：" + elemImage.getPPI());
        }
    }
    
    /**
     * ストロークカラー情報を取得するための関数。
     * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
     * @param pathElement パスエレメントを表現したクラス
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getStrokeColorInfo(PtlEditPath pathElement)
                        throws PtlException, Exception, Error {
        System.out.println("-");
        System.out.println("ストロークカラー情報：");
        try(PtlColorSpace strokeColorSpace = pathElement.getStrokeColorSpace();
            PtlColor strokeColor = pathElement.getStrokeColor()){
            
            PtlColorSpace.COLOR_SPACE_TYPE colorSpaceType =strokeColorSpace.getType();
            switch(colorSpaceType){
            case TYPE_DEVICECMYK:
                try(PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)strokeColor) {
                    getCMYKInfo(deviceCMYK);
                }
                break;
                
            case TYPE_DEVICEGRAY:
                try(PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)strokeColor) {
                    getGrayInfo(deviceGray);
                }
                break;
            
            case TYPE_DEVICERGB:
                try(PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)strokeColor) {
                    getRGBInfo(deviceRGB);
                }
                break;
                
            default:
                System.out.println("ストロークカラーは"
                    + "読み取れるカラースペースタイプではありませんでした。");
                System.out.println("カラースペースタイプ : " + colorSpaceType);
                break;
            }
        }
            
    }
    /**
     * ストロークカラー情報を取得するための関数。
     * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
     * @param item PtlEditTextItem テキストアイテムを表現したクラス
     * @throws PtlException 
     * @throws Exception 
     * @throws Error  
     */
    private static void getStrokeColorInfo(PtlEditTextItem item)
                        throws PtlException, Exception, Error {
        System.out.println("-");
        System.out.println("ストロークカラー情報：");

        try(PtlColor strokeColor = item.getStrokeColor()) {
            switch(strokeColor.getType()) {
                case TYPE_DEVICE_CMYK:
                    try(PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)item.getStrokeColor()) {
                        getCMYKInfo(deviceCMYK);
                    }
                    break;

                case TYPE_DEVICE_GRAY:
                    try(PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)item.getStrokeColor()) {
                        getGrayInfo(deviceGray);
                    }
                    break;

                case TYPE_DEVICE_RGB:
                    try(PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)item.getStrokeColor()) {
                        getRGBInfo(deviceRGB);
                    }
                    break;

                default:
                    System.out.println("ストロークカラーは"
                                     + "読み取れるカラースペースタイプではありませんでした。");
                    System.out.println("カラースペースタイプ : " + strokeColor.getCSType());
                    break;
            }
        }
    }

    /**
     * 塗りつぶしカラー情報を取得するための関数。
     * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
     * @param pathElement パスエレメントを表現したクラス
     * @throws PtlException 
     * @throws Exception 
     * @throws Error  
     */
    private static void getFillColorInfo(PtlEditPath pathElement)
                        throws PtlException, Exception, Error {
        System.out.println("-");
        System.out.println("塗りつぶしカラー情報：");
        try(PtlColorSpace fillColorSpace = pathElement.getStrokeColorSpace();
            PtlColor fillColor = pathElement.getStrokeColor()){
            
            PtlColorSpace.COLOR_SPACE_TYPE colorSpaceType =fillColorSpace.getType();
            switch(colorSpaceType) {
            case TYPE_DEVICECMYK:
                try(PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)fillColor) {
                    getCMYKInfo(deviceCMYK);
                }
                break;
                            
            case TYPE_DEVICEGRAY:
                try(PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)fillColor) {
                    getGrayInfo(deviceGray);
                }
                break;

            case TYPE_DEVICERGB:
                try(PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)fillColor) {
                    getRGBInfo(deviceRGB);
                }
                break;
            
            default:
                System.out.println("ストロークカラーは"
                    + "読み取れるカラースペースタイプではありませんでした。");
                System.out.println("カラースペースタイプ : " + colorSpaceType);
                break;
            }
        }
            
    }
    /**
     * 塗りつぶしカラー情報を取得するための関数。
     * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
     * @param item PtlEditTextItem テキストアイテムを表現したクラス
     * @throws PtlException 
     * @throws Exception 
     * @throws Error  
     */
    private static void getFillColorInfo(PtlEditTextItem item)
                        throws PtlException, Exception, Error {
        System.out.println("-");
        System.out.println("塗りつぶしカラー情報：");
        
        try(PtlColor fillColor = item.getFillColor()) {
            switch(fillColor.getType()) {
                case TYPE_DEVICE_CMYK:
                    try(PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)item.getFillColor()) {
                        getCMYKInfo(deviceCMYK);
                    }
                    break;

                case TYPE_DEVICE_GRAY:
                    try(PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)item.getFillColor()) {
                        getGrayInfo(deviceGray);
                    }
                    break;

                case TYPE_DEVICE_RGB:
                    try(PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)item.getFillColor()) {
                        getRGBInfo(deviceRGB);
                    }
                    break;

                default:
                    System.out.println("塗りつぶしカラーは"
                                     + "読み取れるカラースペースタイプではありませんでした。");
                    System.out.println("カラースペースタイプ : " + fillColor.getCSType());
                    break;
            }
        }
    }    
    /**
     * deviceCYMKの各要素の数値を表示する関数です
     * @param cmyk
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getCMYKInfo(PtlColorDeviceCMYK cmyk)
                        throws PtlException, Exception, Error {
        float paramC = cmyk.getC();
        float paramM = cmyk.getM();
        float paramY = cmyk.getY();
        float paramK = cmyk.getK();
        
        System.out.println("DeviceCMYK色の要素[C, M, Y, K]の値は、");
        System.out.println("[" + paramC + ", " + paramM + ", " +
                                 paramY + ", " + paramK + "]です。");
    }
    
    /**
     * deviceGrayの各要素の数値を表示する関数です
     * @param gray
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getGrayInfo(PtlColorDeviceGray gray)
                        throws PtlException, Exception, Error {
        float paramGray = gray.getGray();
        
        System.out.println("DeviceGray色のGray要素の値は、");
        System.out.println("[" + paramGray + "]です。");
    }

    /**
     * deviceRGBの各要素の数値を表示する関数です
     * @param rgb
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getRGBInfo(PtlColorDeviceRGB rgb)
                        throws PtlException, Exception, Error {
        float paramR = rgb.getR();
        float paramG = rgb.getG();
        float paramB = rgb.getB();
        
        System.out.println("DeviceRGB色の[R, G, B]の要素の値は、");
        System.out.println("[" + paramR + ", " + paramG +", " + 
                                 paramB + "]です。");
    }
    
    /**
     * PtlRectの各情報を表示する関数です。
     * @param rect
     * @param rectName
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getRectInfo(PtlRect rect, String rectName)
                        throws PtlException, Exception, Error {
        float top = rect.getTop();
        float bottom = rect.getBottom();
        float left = rect.getLeft();
        float right = rect.getRight();
        System.out.println(rectName + "の[top, bottom, left, right]は、");
        System.out.println("[" + top +", " + bottom +", " + left +", " + right +"]です。"); 
    }

}