/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：エレメントに含まれるPtlEditTextの情報を表示します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class GetPtlEditTextInfo {

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java GetPtlEditTextInfo in-pdf-file");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 1) {
            printUsage();
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                if (pages.isEmpty()) { // ページコンテナが空かどうか
                    System.err.println("入力PDFのページコンテナが空");
                    return;
                }
                
                for(int i = 0;i < pages.getCount();i++) { // ページ数の取得
                    System.out.println("-----------------");
                    System.out.println(i+"ページ目の情報を表示します。");
                    try (PtlPage page = pages.get(i)) {
                        getEditTextInfo(page);
                    }
                    System.out.println("-----------------");
                }
            }
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }


    /**
     * 読み込まれたページに含まれるTEXTエレメントが持つ情報を表示する関数です。
     * @param page 読み込まれたページ
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getEditTextInfo(PtlPage page)
                        throws PtlException, Exception, Error {
        try(PtlContent content = page.getContent();// 挿入先ページコンテントの取得
            PtlEditElements elems = content.getEditElements(PtlContent.GET_TEXT)) {
            int numOfElements = elems.getCount();
            System.out.println("このページに含まれるTEXTエレメントの数は" + numOfElements + "です。");

            for(int i = 0; i < numOfElements; i++) {
                try(PtlEditElement element = elems.get(i)) {
                    System.out.println("---" + i + "番目のTEXTエレメントの情報を表示します。---");


                    //エレメントに共通の情報（BBox, ClippedBox）の情報を表示
                    getElementInfo(element);


                    //タイプがTEXTであることを確認
                    if(element.getType() == PtlEditElement.ELEMENT_TYPE.TYPE_TEXT) {
                        try(PtlEditText textElement = (PtlEditText)element;
                            PtlEditTextItems textItems = textElement.getTextItems()) {

                            System.out.println("TEXTエレメント固有の情報を表示します。");
                            System.out.println("テキストアイテムの情報を表示します。");
                            getTextItemsInfo(textItems);
                        }
                    }
                    else{
                        System.out.println("ERROR: このエレメントはTEXTではありません。");
                        System.out.println("Element Number: " + i);
                        System.out.println("Element Type: " + element.getType());
                    }
                }
            }
        }
    }
    

    /**
     * エレメントが共通して持つ情報を表示する関数です
     * @param element
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getElementInfo(PtlEditElement element)
                        throws PtlException, Exception, Error {
        System.out.println("エレメントのBBoxの情報を表示します。");
        //BBoxの情報を印字
        try(PtlRect bBox = element.getBBox()) {
            getRectInfo(bBox, "このエレメントのBBox");
        }
        
        //ClippedBoxの情報を印字
        if(element.hasClip()){
            try(PtlRect clippedBox = element.getClippedBBox()) {
                System.out.println("このエレメントにはクリッピングがあります。");
                getRectInfo(clippedBox, "このエレメントのClippedBox");
            }
        }
    }
    
    /**
     * Textアイテムが持つ情報を表示する関数です
     * @param textItems そのエレメントが持つTextItemsコンテナクラス
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getTextItemsInfo(PtlEditTextItems textItems)
                        throws PtlException, Exception, Error {
        int numOfTextItems = textItems.getCount();
        System.out.println("このテキストアイテムに含まれるアイテム数は" + numOfTextItems + "です。");
        for(int j = 0; j < numOfTextItems; j++) {
            System.out.println( j + "番目のアイテム情報：");
            System.out.println("---");

            try(PtlEditTextItem item = textItems.get(j);
                PtlRect textItemBBox =item.getBBox();
                PtlFontInfo fontInfo = item.getFontInfo()) {
                System.out.println("テキストアイテムのBBox情報：");
                getRectInfo(textItemBBox, "このテキストアイテムのBBox");

                System.out.println("-");
                System.out.println("テキストアイテムが含むテキスト：");
                System.out.println(item.getText());

                System.out.println("-");
                System.out.println("フォント情報：");
                System.out.println("フォント名：" + fontInfo.getFontName());
                System.out.println("フォントタイプ：" +  fontInfo.getFontType());
                System.out.println("エンコーディングタイプ：" + fontInfo.getEncodingType());
                if(fontInfo.getEncodingType()==PtlFontInfo.ENCODING_TYPE.ENCODING_NAME) {
                    System.out.println("エンコーディング名：" + fontInfo.getEncodingName());
                }
                if(fontInfo.isEmbedded()) {
                    System.out.println("フォント埋め込み：あり");
                }else {
                    System.out.println("フォント埋め込み：なし");
                }

                System.out.println("-");
                System.out.print("ペイントフラグ：");
                int paintFlag = item.getPaintFlags();
                switch(paintFlag) {
                    case PtlEditPath.PAINT_FILL: //パスを塗りつぶす
                        System.out.println("パスを塗りつぶす");
                        System.out.println("塗りつぶしカラー情報を取得します。");
                        getFillColorInfo(item); //塗りつぶしカラー情報を取得するための関数
                        break;

                    case PtlEditPath.PAINT_INVISIBLE: //ストロークも塗りつぶしも行いません。
                        System.out.println("ストロークも塗りつぶしも行わない");
                        break;

                    case PtlEditPath.PAINT_STROKE: //ストロークする
                        System.out.println("ストロークする");
                        System.out.println("ストロークカラー情報を取得します");
                        getStrokeColorInfo(item); //ストロークカラー情報を取得するための関数
                        break;

                    default:
                        System.out.println("不明なフラグが取得されました。");
                        System.out.println("paintFlag = " + paintFlag);
                        break;
                }
                System.out.println("---");
            }
        }
    }
    
    
    
    /**
     * ストロークカラー情報を取得するための関数。
     * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
     * @param item PtlEditTextItem テキストアイテムを表現したクラス
     * @throws PtlException 
     * @throws Exception 
     * @throws Error  
     */
    private static void getStrokeColorInfo(PtlEditTextItem item)
                        throws PtlException, Exception, Error {
        System.out.println("-");
        System.out.println("ストロークカラー情報：");

        try(PtlColor strokeColor = item.getStrokeColor()) {
            switch(strokeColor.getType()) {
                case TYPE_DEVICE_CMYK:
                    try(PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)item.getStrokeColor()) {
                        getCMYKInfo(deviceCMYK);
                    }
                    break;

                case TYPE_DEVICE_GRAY:
                    try(PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)item.getStrokeColor()) {
                        getGrayInfo(deviceGray);
                    }
                    break;

                case TYPE_DEVICE_RGB:
                    try(PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)item.getStrokeColor()) {
                        getRGBInfo(deviceRGB);
                    }
                    break;

                default:
                    System.out.println("ストロークカラーは"
                                     + "読み取れるカラースペースタイプではありませんでした。");
                    System.out.println("カラースペースタイプ : " + strokeColor.getCSType());
                    break;
            }
        }
    }

    /**
     * 塗りつぶしカラー情報を取得するための関数。
     * デバイスカラー情報に応じて色情報を取り出すための関数を呼び出す
     * @param item PtlEditTextItem テキストアイテムを表現したクラス
     * @throws PtlException 
     * @throws Exception 
     * @throws Error  
     */
    private static void getFillColorInfo(PtlEditTextItem item)
                        throws PtlException, Exception, Error {
        System.out.println("-");
        System.out.println("塗りつぶしカラー情報：");
        
        try(PtlColor fillColor = item.getFillColor()) {
            switch(fillColor.getType()) {
                case TYPE_DEVICE_CMYK:
                    try(PtlColorDeviceCMYK deviceCMYK = (PtlColorDeviceCMYK)item.getFillColor()) {
                        getCMYKInfo(deviceCMYK);
                    }
                    break;

                case TYPE_DEVICE_GRAY:
                    try(PtlColorDeviceGray deviceGray = (PtlColorDeviceGray)item.getFillColor()) {
                        getGrayInfo(deviceGray);
                    }
                    break;

                case TYPE_DEVICE_RGB:
                    try(PtlColorDeviceRGB deviceRGB = (PtlColorDeviceRGB)item.getFillColor()) {
                        getRGBInfo(deviceRGB);
                    }
                    break;

                default:
                    System.out.println("塗りつぶしカラーは"
                                     + "読み取れるカラースペースタイプではありませんでした。");
                    System.out.println("カラースペースタイプ : " + fillColor.getCSType());
                    break;
            }
        }
    }
    
    /**
     * deviceCYMKの各要素の数値を表示する関数です
     * @param cmyk
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getCMYKInfo(PtlColorDeviceCMYK cmyk)
                        throws PtlException, Exception, Error {
        float paramC = cmyk.getC();
        float paramM = cmyk.getM();
        float paramY = cmyk.getY();
        float paramK = cmyk.getK();
        
        System.out.println("DeviceCMYK色の要素[C, M, Y, K]の値は、");
        System.out.println("[" + paramC + ", " + paramM + ", " +
                                 paramY + ", " + paramK + "]です。");
    }
    
    /**
     * deviceGrayの各要素の数値を表示する関数です
     * @param gray
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getGrayInfo(PtlColorDeviceGray gray)
                        throws PtlException, Exception, Error {
        float paramGray = gray.getGray();
        
        System.out.println("DeviceGray色のGray要素の値は、");
        System.out.println("[" + paramGray + "]です。");
    }

    /**
     * deviceRGBの各要素の数値を表示する関数です
     * @param rgb
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getRGBInfo(PtlColorDeviceRGB rgb)
                        throws PtlException, Exception, Error {
        float paramR = rgb.getR();
        float paramG = rgb.getG();
        float paramB = rgb.getB();
        
        System.out.println("DeviceRGB色の[R, G, B]の要素の値は、");
        System.out.println("[" + paramR + ", " + paramG +", " + 
                                 paramB + "]です。");
    }
    
    /**
     * PtlRectの各情報を表示する関数です。
     * @param rect
     * @param rectName
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    private static void getRectInfo(PtlRect rect, String rectName)
                        throws PtlException, Exception, Error {
        float top = rect.getTop();
        float bottom = rect.getBottom();
        float left = rect.getLeft();
        float right = rect.getRight();
        System.out.println(rectName + "の[top, bottom, left, right]は、");
        System.out.println("[" + top +", " + bottom +", " + left +", " + right +"]です。"); 
    }

}