/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：FDFに保存されたフォームフィールドデータを指定したPDFにインポートします。
        その後そのPDFを指定した名前で出力します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class ImportFormFieldsFromFDF {
    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.print("usage: java ImportFormFieldsFromFDF in-pdf-file");
        System.out.println(" out-pdf-file fdf-file-to-import");
    }

    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage();
            return;
        }

        // コマンドライン引数の取得
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlParamInput FdfToImport = new PtlParamInput(args[2]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);

            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.out.println("ページコンテナが空です。処理を終了します。");
                    return;
                }
            }
            // FDFファイルからフォームフィールドデータをインポートする
            //（設定できなかったフォームフィールドの値を取得する）
            try(PtlFormFieldValues failedFormFieldValues = doc.importFormFieldsFromFDF(FdfToImport)) {
                getFailedFormFieldInfo(failedFormFieldValues);
                System.out.println("フォームフィールドデータをインポートしました。");
            }
            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }
    
    
    private static void getFailedFormFieldInfo(PtlFormFieldValues formFieldValues)
                        throws PtlException, Exception, Error {
        int numOfFormFieldValues = formFieldValues.getCount();
        System.out.println("設定できなかったフォームフィールドの数は "
                            + numOfFormFieldValues
                            + " 個です。");
        //0個だった場合は処理を抜ける。
        if(numOfFormFieldValues == 0) {
            return;
        }
        
        System.out.println("設定できなかったフォームフィールドの情報を表示します。");
        System.out.println("---");
        
        // 0個でなかった場合は処理を続け、情報を得る
        for(int i = 0; i < numOfFormFieldValues; i++) {
            
            System.out.println("-");
            System.out.println("index No. "  + i);
            try(PtlFormFieldValue formFieldValue = formFieldValues.get(i)) {
                System.out.println("  フォームフィールド名 : "  + formFieldValue.getFieldName());
                System.out.println("  フォームフィールド値 : "  + formFieldValue.getValue());
            }
        }
        System.out.println("---");
    }
}