/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：テキストボックスを用いた描画の際に、背景色・輪郭色を指定します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;


public class  TBoxDrawTextBoxSetBoxColors{

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java TBoxDrawTextBoxSetBoxColors"
                + " in-pdf-file out-pdf-file"
                + " page-num text-to-add text-align"
                + " back-color-R back-color-G back-color-B"
                + " outline-color-R outline-color-G outline-color-B");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 11) {
            printUsage();
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            
            // コマンドライン引数の読み取り・判定
            int pageToAdd = Integer.parseInt(args[2]);
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);
            if((numPages < 0)||(numPages < pageToAdd)) {
                System.err.println("テキスト挿入ページは全ページ数よりも小さい正の値を指定してください。");
                return;
            }
            String textToAdd = args[3];
            String textAlign = args[4];

            // RGBの数値の取得
            float backR = Float.parseFloat(args[5]);
            float backG = Float.parseFloat(args[6]);
            float backB = Float.parseFloat(args[7]);
            float outlineR = Float.parseFloat(args[8]);
            float outlineG = Float.parseFloat(args[9]);
            float outlineB = Float.parseFloat(args[10]);
            
            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("ページコンテナが空");
                    return;
                }
                
                try (PtlPage page = pages.get(pageToAdd - 1);// ページの取得(パラメータindexは0が先頭のため1を引く)
                     PtlContent content = page.getContent();// ページコンテントの取得
                     PtlRect outputRect = new PtlRect();// 出力矩形の初期化
                     PtlTextBox textBox = content.drawTextBox(setRectCoordinate(br, outputRect), PtlContent.ALIGN.valueOf(textAlign), 210, 297);
                     PtlParamWriteStringTextBox plainParam = new PtlParamWriteStringTextBox(); // 文字描画のパラメータクラス。
                     PtlColorDeviceRGB backColor = new PtlColorDeviceRGB(backR, backG, backB);
                     PtlColorDeviceRGB outlineColor = new PtlColorDeviceRGB(outlineR, outlineG, outlineB)) {
                    // 矩形の背景色の指定
                    textBox.setBackColor(backColor);
                    
                    // 矩形の縁取りの色の指定
                    textBox.setOutlineColor(outlineColor);
                    
                    // 文字列出力
                    textBox.writeString(textToAdd, plainParam);
                    textBox.terminate();
                }
                catch (IllegalArgumentException ex) {//PtlContent.ALIGN指定のエラー処理
                    System.out.println(ex.getMessage());
                    System.out.println("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                    ex.printStackTrace();
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }catch (NumberFormatException numfe) {
                System.out.println("RGBの指定には数値を使用して下さい。");
                System.out.println(numfe.getMessage());
                numfe.printStackTrace();
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    /**
     * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
     * 原点はPDFの左下端。
     * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * @param br BufferedReader。数値の読み取りに使う。
     * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
     * @return 指定したleft, bottom, right, topの数値を持つPtlRect
     * @throws IOException
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    public static PtlRect setRectCoordinate(BufferedReader br, PtlRect outputRect)
           throws IOException, PtlException, Exception, Error {
        float top, bottom, left, right;
        boolean isValueOkay = false;
        
        while(!isValueOkay) {
            System.out.println("指定する矩形の各数値を入力してください。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) {//不正矩形は再入力させる
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            if(right < left) {//不正矩形は再入力させる
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setLeft(left);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setTop(top);
        }
        return outputRect;
    }
}