/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：テキストボックスを用いて指定したページの指定した位置に
        テキストを挿入します。
  　　　配置矩形のみ、プログラム内で指定します。
  　　　行間の大きさ、文字間隔の大きさを数値で指定します。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import java.io.*;
import jp.co.antenna.ptl.*;


public class  TBoxDrawWithSpacing{

    // そのクラスのusageを表示する関数
    private static void printUsage() {
        System.out.println("usage: java TBoxDrawWithSpacing in-pdf-file out-pdf-file"
                + " page-num text-to-add text-align"
                + " line-spacing-num char-spacing-num");
        System.out.println("line-spacing-num : 行間の設定。フォントサイズに対する倍数で、"
                         + "-1～3の値で設定（小数可）");
        System.out.println("char-spacing-num : 文字間隔の設定。フォントサイズに対する倍数で、"
                         + "-1～2の値で設定（小数可）");
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 7) {
            printUsage();
            return;

        }
        
        //line-spacing-num、char-spacing-numの引数の判定
        
        float lineSpacingNum = Float.parseFloat(args[5]);
        if(lineSpacingNum < -1 || 3 < lineSpacingNum ){
            System.out.println("ERROR:line-spacing-numは-1～3の数値で設定して下さい。");
            printUsage(); // usageメッセージの表示
            return;
        }

        float charSpacingNum = Float.parseFloat(args[6]);
        if(charSpacingNum < -1 || 2 < charSpacingNum ){
            System.out.println("ERROR:char-spacing-numは-1～2の数値で設定して下さい。");
            printUsage(); // usageメッセージの表示
            return;
        }
        
        
        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument();
             BufferedReader br = new BufferedReader(new InputStreamReader(System.in))) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            int pageToAdd = Integer.parseInt(args[2]);
            int numPages = doc.getPageCount();
            System.out.println("ページ数：" + numPages);
            if((numPages < 0)||(numPages < pageToAdd)) {
                System.err.println("テキスト挿入ページは全ページ数よりも小さい正の値を指定してください。");
                return;
            }

            
            //引数の読み取り
            String textToAdd = args[3];
            String textAlign = args[4];
            
            try (PtlPages pages = doc.getPages()) {//ページコンテナの取得
                // ページコンテナが空かどうか
                if (pages.isEmpty()) {
                    System.err.println("ページコンテナが空");
                    return;
                }
                
                try (PtlPage page = pages.get(pageToAdd - 1);// ページの取得(パラメータindexは0が先頭のため1を引く)
                     PtlContent content = page.getContent();// ページコンテントの取得
                     PtlRect outputRect = new PtlRect();// 出力矩形の初期化
                     PtlTextBox textBox = content.drawTextBox(setRectCoordinate(br, outputRect), PtlContent.ALIGN.valueOf(textAlign), 210, 297);
                     PtlParamWriteStringTextBox paramSpacing = new PtlParamWriteStringTextBox()) {// 文字描画のパラメータクラス。

                    paramSpacing.setLineSpacing(lineSpacingNum); //行間の設定
                    paramSpacing.setCharSpacing(charSpacingNum); //文字間隔の設定

                    // 文字列出力
                    textBox.writeString(textToAdd, paramSpacing);
                    textBox.terminate();
                }
                catch (IllegalArgumentException ex) {//PtlContent.ALIGN指定のエラー処理
                    System.out.println(ex.getMessage());
                    System.out.println("ERROR : alignにはPtlContent.ALIGNに含まれる名前を指定してください。");
                    ex.printStackTrace();
                }
            }

            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    /**
     * 矩形の各値を入力してその座標値をもつ矩形を返す関数。
     * 原点はPDFの左下端。
     * bottomよりtopが大きい、leftよりもrightが大きいなどの矛盾した数値は入力できない。
     * 特に外部からの呼び出しを想定しないためprivateとする。
     * @param br BufferedReader。数値の読み取りに使う。
     * @param outputRect left, bottom, right, topの数値を入力する対象のPtlRect
     * @return 指定したleft, bottom, right, topの数値を持つPtlRect
     * @throws IOException
     * @throws PtlException
     * @throws Exception
     * @throws Error 
     */
    public static PtlRect setRectCoordinate(BufferedReader br, PtlRect outputRect)
           throws IOException, PtlException, Exception, Error {
        float top, bottom, left, right;
        boolean isValueOkay = false;
        
        while(!isValueOkay) {
            System.out.println("指定する矩形の各数値を入力してください。");
            System.out.print("top (mm) : ");
            top = Float.parseFloat(br.readLine());
            System.out.print("bottom (mm) : ");
            bottom = Float.parseFloat(br.readLine());
            if(top < bottom) {//不正矩形は再入力させる
                System.out.println("topの値はbottomよりも大きい値を指定して再度入力してください。");
                continue;
            }
            System.out.print("left (mm) : ");
            left = Float.parseFloat(br.readLine());
            System.out.print("right (mm) : ");
            right = Float.parseFloat(br.readLine());
            if(right < left) {//不正矩形は再入力させる
                System.out.println("rightの値はleftよりも大きい値を指定して再度入力してください。");
                continue;
            }

            //矩形を正しく指定できた場合の処理
            isValueOkay = true;
            outputRect.setLeft(left);
            outputRect.setBottom(bottom);
            outputRect.setRight(right);
            outputRect.setTop(top);
        }
        return outputRect;
    }
}