/*
  Antenna House PDF Tool API V7.0
  Java Interface sample program

  概要：透かしに名前を付けます。

  Copyright (C) 2023- Antenna House, Inc. All rights reserved.
  Antenna House is a trademark of [Antenna House, Inc.]

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
*/

package cookbook;

import jp.co.antenna.ptl.*;

public class WaterMarkSetAcrobatCompatible {
    // そのクラスのusageを表示する関数
    public static void printUsage() {
        System.out.print("usage: java WaterMarkSetName in-pdf-file out-pdf-file ");
        System.out.println("set-acrobat-compatible");
        System.out.println("set-acrobat-compatible : ");
        System.out.println("0:透かしをAcrobat互換にしない  1:透かしをAcrobat互換にする");        
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        if (args.length < 3) {
            printUsage(); // usageメッセージの表示
            return;
        }

        // コマンドライン引数の読み取り
        boolean setAcrobatCompatible = false;
        try {
            setAcrobatCompatible = readBoolArgs(args[2],
                                                "set-acrobat-compatibleは 0か1で指定してください。");
        }
        catch (IllegalArgumentException ex) {
            System.out.println(ex.getMessage());
            printUsage(); // usageメッセージの表示
            return;
        }

        try (PtlParamInput inputFile = new PtlParamInput(args[0]);
             PtlParamOutput outputFile = new PtlParamOutput(args[1]);
             PtlPDFDocument doc = new PtlPDFDocument()) {
            // PDFファイルをロードします。
            doc.load(inputFile);
            // 透かしの追加
            appendWaterMarkSetAcrobatCompatible(doc, setAcrobatCompatible);
            // ファイルに保存します。
            doc.save(outputFile);
        }
        catch (PtlException pex) {
            System.out.println("PtlException : ErrorCode = " + pex.getErrorCode() +
                               "\n  " + pex.getErrorMessage());
        }
        catch (Exception | Error ex) {
            System.out.println(ex.getMessage());
            ex.printStackTrace();
        }
        finally {
            System.out.println("-- 完了 --");
        }
    }

    public static void appendWaterMarkSetAcrobatCompatible(PtlPDFDocument doc,
                                                           boolean setAcrobatCompatible)
        throws PtlException, Exception, Error {
        try (PtlParamWaterMarkText waterMarkText = new PtlParamWaterMarkText()) {
            // 透かしの名前の設定
            waterMarkText.setAcrobatCompatible(setAcrobatCompatible);

            //setAcrobatCompatibleをtrueにしたときにPDF出力ができるため、以下の設定をする
            // PtlParamWaterMark.setCustomPageRange()では不連続なページを指定しない
            // PtlParamWaterMark.setMargin()を設定しない
            // PtlParamWaterMark.setRect()を設定しない
            // PtlParamWaterMark.setTiling()をfalseにする
            waterMarkText.setTiling(false);
            // PtlParamWaterMarkText.setOutlineColor()を設定しない
            // PtlParamWaterMarkText.setWriteDiagonal()をfalseにする
            waterMarkText.setWriteDiagonal(false);
            
            
            // 以下、その他の項目について透かしであることが分かりやすいようデフォルトの設定をする
            // 透かしの名前の設定
            waterMarkText.setName("waterMarkText");
            // 透かしの配置の設定
            waterMarkText.setAlign(PtlParamWaterMark.ALIGN.ALIGN_CENTER);
            // 透かしのZオーダーの設定 ZORDER_FRONT = 1 
            waterMarkText.setZorder(PtlParamWaterMark.ZORDER.ZORDER_FRONT);
            // 透かしの不透明度の設定
            waterMarkText.setOpacity(1.0f);
            // 透かしに指定する文字列の設定
            waterMarkText.setString("WaterMarkText test");
            // 透かしに指定するフォントの設定
            try (PtlParamFont font = new PtlParamFont("Times New Roman", 72.0f,
                                                      PtlParamFont.WEIGHT.WEIGHT_MEDIUM, false, true)) {
                waterMarkText.setFont(font);
            }
            // 透かしの文字に指定する色の設定
            try (PtlColorDeviceRGB colorText = new PtlColorDeviceRGB(1.0f, 0.0f, 0.0f)) {
                waterMarkText.setTextColor(colorText);
            }
            // 透かしのテキストを任意の角度で配置する設定
            waterMarkText.setTextAngle(15.0f);
            // デフォルトの設定終わり

            // 透かしを文書に設定する
            doc.appendWaterMark(waterMarkText);
        }
    }
    
    /**
     * 0または1を入力されたargsにより、trueまたはfalseを返すメソッド。
     * 
     * @param args 与えられるコマンドライン引数。0または1でtrueまたはfalseを指定する。
     * @param errorMessage argsが0か1でなかった場合に出力されるエラーメッセージを指定する。
     * @return argsの数値を読み取った結果を戻す
     * @throws java.lang.IllegalArgumentException argsが0か1でなかった場合に発生。
     */
    public static boolean readBoolArgs(String args, String errorMessage)
        throws IllegalArgumentException {
        boolean trueOrFalse = false;
        
        // argsの読み取り
        switch(args) {
        case "0":
            trueOrFalse = false;
            break;
        case "1":
            trueOrFalse = true;
            break;
        default:
            throw new IllegalArgumentException(errorMessage);
        }

        return trueOrFalse;
    }
}